<?php

declare(strict_types=1);

namespace Laminas\ApiTools\MvcAuth\Authentication;

use Laminas\ApiTools\MvcAuth\Identity;
use Laminas\ApiTools\MvcAuth\MvcAuthEvent;
use Laminas\Http\Request;
use Laminas\Http\Response;
use OAuth2\Request as OAuth2Request;
use OAuth2\Response as OAuth2Response;
use OAuth2\Server as OAuth2Server;

use function in_array;
use function is_array;
use function is_string;
use function method_exists;

class OAuth2Adapter extends AbstractAdapter
{
    /**
     * Authorization header token types this adapter can fulfill.
     *
     * @var array
     */
    protected $authorizationTokenTypes = ['bearer'];

    /** @var OAuth2Server */
    private $oauth2Server;

    /**
     * Authentication types this adapter provides.
     *
     * @var array
     */
    private $providesTypes = ['oauth2'];

    /**
     * Request methods that will not have request bodies
     *
     * @var array
     */
    private $requestsWithoutBodies = [
        'GET',
        'HEAD',
        'OPTIONS',
    ];

    /** @psalm-param null|string|string[] $types */
    public function __construct(OAuth2Server $oauth2Server, $types = null)
    {
        $this->oauth2Server = $oauth2Server;

        if (is_string($types) && ! empty($types)) {
            $types = [$types];
        }

        if (is_array($types)) {
            $this->providesTypes = $types;
        }
    }

    /**
     * @return array Array of types this adapter can handle.
     */
    public function provides()
    {
        return $this->providesTypes;
    }

    /**
     * Attempt to match a requested authentication type
     * against what the adapter provides.
     *
     * @param string $type
     * @return bool
     */
    public function matches($type)
    {
        return in_array($type, $this->providesTypes, true);
    }

    /**
     * Determine if the given request is a type (oauth2) that we recognize
     *
     * @return false|string
     */
    public function getTypeFromRequest(Request $request)
    {
        $type = parent::getTypeFromRequest($request);

        if (false !== $type) {
            return 'oauth2';
        }

        if (
            ! in_array($request->getMethod(), $this->requestsWithoutBodies)
            && $request->getHeaders()->has('Content-Type')
            && $request->getHeaders()->get('Content-Type')->match('application/x-www-form-urlencoded')
            && $request->getPost('access_token')
        ) {
            return 'oauth2';
        }

        if (null !== $request->getQuery('access_token')) {
            return 'oauth2';
        }

        return false;
    }

    /**
     * Perform pre-flight authentication operations.
     *
     * Performs a no-op; nothing needs to happen for this adapter.
     *
     * @return void
     */
    public function preAuth(Request $request, Response $response)
    {
    }

    /**
     * Attempt to authenticate the current request.
     *
     * @return Identity\AuthenticatedIdentity|Identity\GuestIdentity|Response
     */
    public function authenticate(Request $request, Response $response, MvcAuthEvent $mvcAuthEvent)
    {
        $oauth2request = new OAuth2Request(
            $request->getQuery()->toArray(),
            $request->getPost()->toArray(),
            [],
            $request->getCookie() ? $request->getCookie()->getArrayCopy() : [],
            $request->getFiles() ? $request->getFiles()->toArray() : [],
            method_exists($request, 'getServer') ? $request->getServer()->toArray() : $_SERVER,
            $request->getContent(),
            $request->getHeaders()->toArray()
        );

        $token = $this->oauth2Server->getAccessTokenData($oauth2request);

        // Failure to validate
        if (! $token) {
            return $this->processInvalidToken($response);
        }

        $identity = new Identity\AuthenticatedIdentity($token);
        $identity->setName($token['user_id']);

        return $identity;
    }

    /**
     * Handle a invalid Token.
     *
     * @return Response|Identity\GuestIdentity
     */
    private function processInvalidToken(Response $response)
    {
        $oauth2Response = $this->oauth2Server->getResponse();
        $status         = $oauth2Response->getStatusCode();

        // 401 or 403 mean invalid credentials or unauthorized scopes; report those.
        if (in_array($status, [401, 403], true) && null !== $oauth2Response->getParameter('error')) {
            return $this->mergeOAuth2Response($status, $response, $oauth2Response);
        }

        // Merge in any headers; typically sets a WWW-Authenticate header.
        $this->mergeOAuth2ResponseHeaders($response, $oauth2Response->getHttpHeaders());

        // Otherwise, no credentials were present at all, so we just return a guest identity.
        return new Identity\GuestIdentity();
    }

    /**
     * Merge the OAuth2\Response instance's status and headers into the current Laminas\Http\Response.
     *
     * @param int $status
     * @return Response
     */
    private function mergeOAuth2Response($status, Response $response, OAuth2Response $oauth2Response)
    {
        $response->setStatusCode($status);
        return $this->mergeOAuth2ResponseHeaders($response, $oauth2Response->getHttpHeaders());
    }

    /**
     * Merge the OAuth2\Response headers into the current Laminas\Http\Response.
     *
     * @param array $oauth2Headers
     * @return Response
     */
    private function mergeOAuth2ResponseHeaders(Response $response, array $oauth2Headers)
    {
        if (empty($oauth2Headers)) {
            return $response;
        }

        $headers = $response->getHeaders();
        foreach ($oauth2Headers as $header => $value) {
            $headers->addHeaderLine($header, $value);
        }

        return $response;
    }
}
